/*
  Resilience Command Center
  - No backend. Mock telemetry + scenario engine.
  - Predictive resilience: leading indicators BEFORE failures.

  Capstone topic:
  “Design of Hybrid Quantum–Neural AI Systems for Predictive Cyber Threat Detection and Adversarial Resistance.”
*/

(() => {
  "use strict";

  // -----------------------------
  // Utilities
  // -----------------------------
  const clamp = (v, min, max) => Math.max(min, Math.min(max, v));
  const lerp = (a, b, t) => a + (b - a) * t;
  const pct = (v, min, max) => (clamp(v, min, max) - min) / (max - min);

  function fmtTime(d = new Date()) {
    const hh = String(d.getHours()).padStart(2, "0");
    const mm = String(d.getMinutes()).padStart(2, "0");
    const ss = String(d.getSeconds()).padStart(2, "0");
    return `${hh}:${mm}:${ss}`;
  }

  function weightedJitter(scale = 1) {
    // small correlated noise
    return (Math.random() - Math.random()) * scale;
  }

  // -----------------------------
  // DOM helpers
  // -----------------------------
  const $ = (sel) => document.querySelector(sel);

  function setText(sel, text) {
    const el = $(sel);
    if (el) el.textContent = String(text);
  }

  function setHTML(sel, html) {
    const el = $(sel);
    if (el) el.innerHTML = html;
  }

  function setProgress(sel, value01) {
    const bar = $(sel);
    if (!bar) return;
    const v = clamp(value01, 0, 1);
    bar.style.width = `${Math.round(v * 100)}%`;
    bar.setAttribute("aria-valuenow", String(Math.round(v * 100)));
  }

  function setBadge(sel, level) {
    const el = $(sel);
    if (!el) return;
    const map = {
      Low: "badge rounded-pill bg-success",
      Medium: "badge rounded-pill bg-warning text-dark",
      High: "badge rounded-pill bg-danger",
      Critical: "badge rounded-pill bg-dark",
      Ready: "badge rounded-pill bg-secondary",
      Running: "badge rounded-pill bg-success",
      Blocked: "badge rounded-pill bg-danger",
      STABLE: "badge rounded-pill bg-success",
      WARNING: "badge rounded-pill bg-warning text-dark",
      RISING: "badge rounded-pill bg-danger",
    };
    el.className = map[level] || "badge rounded-pill bg-secondary";
    el.textContent = level;
  }

  function setGauge(sel, value01) {
    const el = $(sel);
    if (!el) return;
    const v = clamp(value01, 0, 1);
    el.style.setProperty("--g", `${Math.round(v * 100)}%`);
  }

  // -----------------------------
  // Sparklines (canvas)
  // -----------------------------
  function drawSparkline(canvas, series, opts = {}) {
    if (!canvas) return;
    const ctx = canvas.getContext("2d");
    if (!ctx) return;

    const w = canvas.width;
    const h = canvas.height;
    ctx.clearRect(0, 0, w, h);

    const pad = 4;
    const min = opts.min ?? Math.min(...series);
    const max = opts.max ?? Math.max(...series);

    // background
    ctx.globalAlpha = 1;
    ctx.fillStyle = "rgba(15, 23, 42, 0.35)";
    ctx.fillRect(0, 0, w, h);

    // grid line
    ctx.strokeStyle = "rgba(148, 163, 184, 0.18)";
    ctx.lineWidth = 1;
    ctx.beginPath();
    ctx.moveTo(0, h - 0.5);
    ctx.lineTo(w, h - 0.5);
    ctx.stroke();

    const n = series.length;
    if (n < 2) return;

    const xStep = (w - pad * 2) / (n - 1);
    ctx.strokeStyle = opts.stroke ?? "rgba(226, 232, 240, 0.9)";
    ctx.lineWidth = 2;

    ctx.beginPath();
    for (let i = 0; i < n; i++) {
      const x = pad + i * xStep;
      const y = h - pad - pct(series[i], min, max) * (h - pad * 2);
      if (i === 0) ctx.moveTo(x, y);
      else ctx.lineTo(x, y);
    }
    ctx.stroke();

    // end dot
    const last = series[n - 1];
    const x = pad + (n - 1) * xStep;
    const y = h - pad - pct(last, min, max) * (h - pad * 2);
    ctx.fillStyle = "rgba(34, 197, 94, 0.95)";
    ctx.beginPath();
    ctx.arc(x, y, 3.2, 0, Math.PI * 2);
    ctx.fill();
  }

  // -----------------------------
  // Scenarios
  // -----------------------------
  const SCENARIOS = {
    normal: {
      name: "Normal Operations",
      desc: "Stable telemetry; leading indicators low; controls effective.",
      dyn: {
        driftTarget: 0.10,
        lagTarget: 120,
        precursorTarget: 0.10,
        advTarget: 0.18,
        costTarget: 1.02,
        fidelityTarget: 0.98,
        qLatencyTarget: 36,
      },
    },
    adversarial: {
      name: "Adversarial Campaign",
      desc: "Attack pressure rises; false positives rise; TTD/TTR degrade unless mitigated.",
      dyn: {
        driftTarget: 0.22,
        lagTarget: 260,
        precursorTarget: 0.26,
        advTarget: 0.68,
        costTarget: 1.15,
        fidelityTarget: 0.96,
        qLatencyTarget: 42,
      },
    },
    quantum: {
      name: "Degraded Quantum Fidelity",
      desc: "Fidelity drops; sampler latency grows; hybrid inference slows; risk increases.",
      dyn: {
        driftTarget: 0.16,
        lagTarget: 210,
        precursorTarget: 0.20,
        advTarget: 0.34,
        costTarget: 1.10,
        fidelityTarget: 0.86,
        qLatencyTarget: 85,
      },
    },
  };

  // -----------------------------
  // State
  // -----------------------------
  const state = {
    scenario: "normal",
    paused: false,

    // leading indicators
    lagMs: 120,
    drift: 0.12,
    precursors: 0.08,
    advPressure: 0.22,
    costSpike: 1.05,
    fidelity: 0.98,

    // ops metrics
    serviceHealth: 0.96,
    p95: 210,
    errRate: 0.6,
    qQueue: 2,
    samplerLatency: 36,
    qfeHealth: 0.91,

    // model metrics
    conf: 0.93,
    fpPer1k: 2.1,
    attackSuccess: 0.07,

    // response
    ttd: 24,
    ttr: 68,

    // governance
    openRisks: 5,
    exceptions: 2,
    evidence: 41,
    lastAuditMin: 12,
    modelVersion: "v1.2.1",

    // mitigations
    quarantine: false,
    rateLimit: false,
    quantumBackoff: false,

    // derived
    riskLevel: "Low",
    riskReason: "Baseline telemetry stable; leading indicators within bounds.",
    resilience: 92,
    retrainStatus: "Ready",
    robustLabel: "STRONG",
    mitre: ["T1059"],
    alerts: { critical: 0, high: 0, medium: 0, low: 0 },

    // history for charts (60s)
    hist: { lag: [], drift: [], prec: [], adv: [], cost: [], fid: [] },
    feed: [],

    charts: { leading: null, ops: null },
    chartWindow: 60,
    chartLabels: [],
  };

  const HIST_MAX = 60;

  function pushHist(key, v) {
    const arr = state.hist[key];
    arr.push(v);
    while (arr.length > HIST_MAX) arr.shift();
  }

  function addEvent(message, severity = "info") {
    const sevIcon = {
      info: "bi-info-circle",
      ok: "bi-check-circle",
      warn: "bi-exclamation-triangle",
      high: "bi-exclamation-octagon",
      crit: "bi-bug",
    }[severity] || "bi-info-circle";

    state.feed.unshift({ t: fmtTime(), message, severity, icon: sevIcon });
    state.feed = state.feed.slice(0, 30);
  }

  // -----------------------------
  // Risk model (explainable)
  // -----------------------------
  function computeRiskAndResilience() {
    // Normalize leading indicators (0..1 bad), except fidelity bad when low
    const lagN = pct(state.lagMs, 100, 650);
    const driftN = pct(state.drift, 0.05, 0.45);
    const precN = pct(state.precursors, 0.05, 0.55);
    const advN = pct(state.advPressure, 0.10, 0.90);
    const costN = pct(state.costSpike, 1.0, 1.35);
    const fidN = 1 - pct(state.fidelity, 0.80, 0.99);

    // Weighted risk
    const leadingRisk =
      0.22 * lagN +
      0.19 * driftN +
      0.18 * precN +
      0.17 * advN +
      0.12 * costN +
      0.12 * fidN;

    // Ops amplification
    const opsAmp =
      0.10 * pct(state.errRate, 0, 5) +
      0.08 * pct(state.p95, 120, 700) +
      0.08 * (1 - state.serviceHealth) +
      0.06 * pct(state.qQueue, 0, 12);

    // Mitigation damping
    const mitigation =
      (state.quarantine ? 0.10 : 0) +
      (state.rateLimit ? 0.08 : 0) +
      (state.quantumBackoff ? 0.06 : 0);

    const effectiveRisk = clamp(leadingRisk + opsAmp - mitigation, 0, 1);

    let level = "Low";
    if (effectiveRisk >= 0.78) level = "Critical";
    else if (effectiveRisk >= 0.58) level = "High";
    else if (effectiveRisk >= 0.35) level = "Medium";

    // Resilience score (0..100) with small bonuses
    const robustnessBonus = clamp((1 - state.attackSuccess) * 6, 0, 6);
    const governanceBonus = clamp((1 - pct(state.openRisks, 0, 20)) * 3, 0, 3);
    const healthBonus = clamp(state.serviceHealth * 4, 0, 4);
    const score = clamp(
      Math.round((1 - effectiveRisk) * 100 + robustnessBonus + governanceBonus + healthBonus),
      0,
      100
    );

    // Explainability: top contributors
    const factors = [
      { k: "Pipeline lag rising", v: lagN },
      { k: "Data drift rising", v: driftN },
      { k: "Anomaly precursors rising", v: precN },
      { k: "Adversarial pressure rising", v: advN },
      { k: "Cost spike trend", v: costN },
      { k: "Quantum fidelity degrading", v: fidN },
    ].sort((a, b) => b.v - a.v);

    state.riskLevel = level;
    state.riskReason = `${factors[0].k}; ${factors[1].k}.`;
    state.resilience = score;

    // Recommendations
    const recs = [];
    if (lagN > 0.45) recs.push("Pipeline lag is rising: check queue/backpressure + shift traffic (leading indicator for missed detections).");
    if (driftN > 0.45) recs.push("Drift sustained: validate feature distributions and prep retraining.");
    if (precN > 0.45) recs.push("Precursors rising: increase sampling + consider quarantine for suspicious hosts.");
    if (advN > 0.45) recs.push("Adversarial campaign likely: enable rate-limit, tighten thresholds, monitor ASR.");
    if (costN > 0.50) recs.push("Cost spike emerging: check noisy alerts and guardrail expensive inference paths.");
    if (fidN > 0.40) recs.push("Quantum fidelity degraded: apply quantum backoff and review hybrid latency split.");
    if (!recs.length) recs.push("Continue monitoring; no immediate action required. Maintain evidence capture cadence.");

    state._recommendations = recs.slice(0, 5);

    // Retrain status
    if (driftN > 0.55 && precN > 0.45) state.retrainStatus = "Ready";
    if (driftN > 0.65 && (state.rateLimit || state.quarantine)) state.retrainStatus = "Running";
    if (advN > 0.75 && !state.rateLimit) state.retrainStatus = "Blocked";

    // Robustness label
    let robust = "STRONG";
    if (state.attackSuccess >= 0.20) robust = "WEAK";
    else if (state.attackSuccess >= 0.12) robust = "FAIR";
    state.robustLabel = robust;

    // MITRE-ish tags
    const tags = [];
    if (advN > 0.5) tags.push("T1566");
    if (precN > 0.45) tags.push("T1059");
    if (lagN > 0.45) tags.push("T1499");
    if (!tags.length) tags.push("T1204");
    state.mitre = tags.slice(0, 3);

    // store
    state._effectiveRisk = effectiveRisk;

    // lag badge status
    const lagBadge = effectiveRisk > 0.58 ? "RISING" : effectiveRisk > 0.35 ? "WARNING" : "STABLE";
    state._lagBadge = lagBadge;

    // trends (simple, based on last 2 points)
    const trend = (arr) => {
      if (arr.length < 2) return "↔";
      const a = arr[arr.length - 2], b = arr[arr.length - 1];
      if (b > a * 1.02) return "↑";
      if (b < a * 0.98) return "↓";
      return "↔";
    };
    state._trendLag = trend(state.hist.lag);
    state._trendDrift = trend(state.hist.drift);
    state._trendAnom = trend(state.hist.prec);
  }

  // -----------------------------
  // Scenario dynamics
  // -----------------------------
  function applyScenarioForces() {
    const s = SCENARIOS[state.scenario].dyn;

    // Drift toward target
    state.drift = clamp(lerp(state.drift, s.driftTarget, 0.08) + weightedJitter(0.01), 0.01, 0.65);

    // Precursors correlate with drift + pressure
    const precTarget = clamp(s.precursorTarget + state.drift * 0.25 + state.advPressure * 0.08, 0.02, 0.75);
    state.precursors = clamp(lerp(state.precursors, precTarget, 0.10) + weightedJitter(0.012), 0.01, 0.90);

    // Lag correlates with precursors and rateLimit mitigation
    const lagTarget = clamp(s.lagTarget + state.precursors * 260 + (state.rateLimit ? -45 : 0), 80, 950);
    state.lagMs = clamp(lerp(state.lagMs, lagTarget, 0.10) + weightedJitter(18), 60, 1200);

    // Pressure (mitigations reduce)
    const advTarget = clamp(s.advTarget - (state.rateLimit ? 0.12 : 0) - (state.quarantine ? 0.08 : 0), 0.05, 0.98);
    state.advPressure = clamp(lerp(state.advPressure, advTarget, 0.10) + weightedJitter(0.02), 0.01, 0.99);

    // Cost spikes correlate with lag
    const costTarget = clamp(s.costTarget + pct(state.lagMs, 80, 650) * 0.18 + (state.rateLimit ? -0.03 : 0), 0.95, 1.6);
    state.costSpike = clamp(lerp(state.costSpike, costTarget, 0.10) + weightedJitter(0.015), 0.90, 1.8);

    // Quantum fidelity + sampler latency
    const fidTarget = clamp(s.fidelityTarget + (state.quantumBackoff ? 0.02 : 0) - pct(state.qQueue, 0, 12) * 0.01, 0.70, 0.995);
    state.fidelity = clamp(lerp(state.fidelity, fidTarget, 0.08) + weightedJitter(0.006), 0.65, 0.999);

    const qLatTarget = clamp(s.qLatencyTarget + (1 - state.fidelity) * 120 + (state.quantumBackoff ? -18 : 0), 18, 220);
    state.samplerLatency = clamp(lerp(state.samplerLatency, qLatTarget, 0.12) + weightedJitter(5), 12, 260);

    // Queue depth
    const qQTarget = clamp(2 + pct(state.samplerLatency, 20, 160) * 8 + (state.quantumBackoff ? -2 : 0), 0, 20);
    state.qQueue = Math.round(clamp(lerp(state.qQueue, qQTarget, 0.18) + weightedJitter(0.8), 0, 30));

    // QFE health
    const qfeTarget = clamp(0.95 - (1 - state.fidelity) * 0.7 - state.qQueue * 0.01, 0.25, 0.99);
    state.qfeHealth = clamp(lerp(state.qfeHealth, qfeTarget, 0.12) + weightedJitter(0.01), 0.1, 0.999);

    // Model confidence & FP
    const confTarget = clamp(0.96 - state.drift * 0.5 - state.advPressure * 0.18 + (state.quarantine ? 0.02 : 0), 0.25, 0.99);
    state.conf = clamp(lerp(state.conf, confTarget, 0.12) + weightedJitter(0.01), 0.05, 0.999);

    const fpTarget = clamp(1.4 + state.drift * 6 + state.advPressure * 2.2 + (state.rateLimit ? -0.3 : 0), 0.4, 20);
    state.fpPer1k = clamp(lerp(state.fpPer1k, fpTarget, 0.16) + weightedJitter(0.25), 0.1, 25);

    // Attack success (ASR)
    const asrTarget = clamp(0.05 + state.advPressure * 0.22 + (state.rateLimit ? -0.03 : 0) + (state.quarantine ? -0.02 : 0), 0.01, 0.55);
    state.attackSuccess = clamp(lerp(state.attackSuccess, asrTarget, 0.12) + weightedJitter(0.01), 0.005, 0.65);

    // CloudOps posture
    const svcTarget = clamp(0.985 - pct(state.lagMs, 120, 650) * 0.10 - pct(state.samplerLatency, 30, 160) * 0.05, 0.72, 0.999);
    state.serviceHealth = clamp(lerp(state.serviceHealth, svcTarget, 0.12) + weightedJitter(0.01), 0.55, 0.999);

    const p95Target = clamp(160 + pct(state.lagMs, 120, 650) * 340 + pct(state.samplerLatency, 30, 160) * 240, 90, 1200);
    state.p95 = clamp(lerp(state.p95, p95Target, 0.12) + weightedJitter(20), 60, 1500);

    const errTarget = clamp(0.3 + pct(state.lagMs, 140, 700) * 1.8 + (state.quarantine ? -0.15 : 0), 0.05, 9.0);
    state.errRate = clamp(lerp(state.errRate, errTarget, 0.16) + weightedJitter(0.08), 0, 12);

    // Response times
    const ttdTarget = clamp(18 + pct(state.advPressure, 0.2, 0.9) * 45 + pct(state.lagMs, 120, 650) * 25 - (state.quarantine ? 6 : 0), 6, 220);
    state.ttd = Math.round(clamp(lerp(state.ttd, ttdTarget, 0.12) + weightedJitter(2), 4, 300));

    const ttrTarget = clamp(55 + pct(state.advPressure, 0.2, 0.9) * 110 + pct(state.lagMs, 120, 650) * 40 - (state.rateLimit ? 10 : 0), 10, 600);
    state.ttr = Math.round(clamp(lerp(state.ttr, ttrTarget, 0.10) + weightedJitter(4), 8, 900));

    // Governance (slow)
    const risksTarget = clamp(5 + pct(state._effectiveRisk ?? 0.2, 0, 1) * 10, 0, 30);
    state.openRisks = Math.round(clamp(lerp(state.openRisks, risksTarget, 0.02) + weightedJitter(0.3), 0, 50));
    const excTarget = clamp(2 + pct(state.advPressure, 0.2, 0.9) * 3, 0, 12);
    state.exceptions = Math.round(clamp(lerp(state.exceptions, excTarget, 0.02) + weightedJitter(0.2), 0, 20));

    // Audit/evidence
    state.lastAuditMin = clamp(state.lastAuditMin + 1 / 60, 0, 999);
    if (Math.random() < 0.03) state.evidence += 1;

    // History
    pushHist("lag", state.lagMs);
    pushHist("drift", state.drift);
    pushHist("prec", state.precursors);
    pushHist("adv", state.advPressure);
    pushHist("cost", state.costSpike);
    pushHist("fid", state.fidelity);

    // Occasional event
    if (Math.random() < 0.05) {
      addEvent(`Telemetry sample: drift=${state.drift.toFixed(2)}, lag=${Math.round(state.lagMs)}ms`, "info");
    }
  }

  function updateAlerts() {
    const r = state._effectiveRisk ?? 0.2;
    const base = Math.round(lerp(1, 12, r));
    const critical = Math.round(base * (r > 0.75 ? 0.35 : 0.12) + (Math.random() < r * 0.2 ? 1 : 0));
    const high = Math.round(base * 0.30 + (Math.random() < r * 0.15 ? 1 : 0));
    const med = Math.round(base * 0.40 + (Math.random() < r * 0.1 ? 1 : 0));
    const low = Math.max(0, base - critical - high - med);

    state.alerts = { critical, high, medium: med, low };

    if (critical > 0 && Math.random() < 0.25) addEvent("Critical alert: pipeline lag + precursors indicate potential misses.", "crit");
    else if (high > 0 && Math.random() < 0.18) addEvent(`High alert mapped to ${state.mitre[0]}.`, "high");
    else if (med > 0 && Math.random() < 0.12) addEvent("Medium alert: drift/precursor anomaly flagged.", "warn");
  }

  // -----------------------------
  // Real-time Chart.js charts
  // -----------------------------
  function initCharts() {
    if (!window.Chart) return;

    const cLead = $("#chartLeading");
    const cOps = $("#chartOps");
    if (!cLead || !cOps) return;

    // Professional dark-theme palette
    const palette = {
      lag: "rgba(56, 189, 248, 0.95)",     // sky
      lagFill: "rgba(56, 189, 248, 0.12)",
      drift: "rgba(167, 139, 250, 0.95)",  // violet
      driftFill: "rgba(167, 139, 250, 0.12)",
      prec: "rgba(245, 158, 11, 0.95)",    // amber
      precFill: "rgba(245, 158, 11, 0.10)",
      score: "rgba(34, 197, 94, 0.95)",    // green
      scoreFill: "rgba(34, 197, 94, 0.10)",
      asr: "rgba(239, 68, 68, 0.95)",      // red
      asrFill: "rgba(239, 68, 68, 0.08)",
      err: "rgba(234, 179, 8, 0.95)",      // yellow
      errFill: "rgba(234, 179, 8, 0.08)",
      grid: "rgba(148,163,184,0.12)",
      tick: "rgba(148,163,184,0.95)",
      label: "rgba(226,232,240,0.95)",
    };

    const commonOpts = {
      responsive: true,
      maintainAspectRatio: false,
      animation: false,
      interaction: { mode: "index", intersect: false },
      plugins: {
        legend: {
          position: "bottom",
          labels: { color: palette.label, boxWidth: 10, boxHeight: 10, usePointStyle: true, pointStyle: "line" }
        },
        tooltip: { enabled: true }
      },
      scales: {
        x: {
          ticks: { color: palette.tick, maxRotation: 0, autoSkip: true, maxTicksLimit: 6 },
          grid: { color: palette.grid }
        }
      },
      elements: { line: { borderWidth: 2 }, point: { radius: 0, hitRadius: 10 } }
    };

    state.charts.leading = new window.Chart(cLead.getContext("2d"), {
      type: "line",
      data: {
        labels: [],
        datasets: [
          { label: "Pipeline Lag (ms)", data: [], yAxisID: "yLag", tension: 0.25, borderColor: palette.lag, backgroundColor: palette.lagFill, fill: true },
          { label: "Drift", data: [], yAxisID: "y01", tension: 0.25, borderColor: palette.drift, backgroundColor: palette.driftFill, fill: true },
          { label: "Precursors", data: [], yAxisID: "y01", tension: 0.25, borderColor: palette.prec, backgroundColor: palette.precFill, fill: true },
        ],
      },
      options: {
        ...commonOpts,
        scales: {
          ...commonOpts.scales,
          yLag: { position: "left", ticks: { color: palette.tick }, grid: { color: palette.grid } },
          y01:  { position: "right", min: 0, max: 1, ticks: { color: palette.tick }, grid: { drawOnChartArea: false } }
        }
      }
    });

    state.charts.ops = new window.Chart(cOps.getContext("2d"), {
      type: "line",
      data: {
        labels: [],
        datasets: [
          { label: "Resilience Score", data: [], yAxisID: "yScore", tension: 0.25, borderColor: palette.score, backgroundColor: palette.scoreFill, fill: true },
          { label: "Attack Success (ASR)", data: [], yAxisID: "y01", tension: 0.25, borderColor: palette.asr, backgroundColor: palette.asrFill, fill: true },
          { label: "Error Rate (%)", data: [], yAxisID: "yErr", tension: 0.25, borderColor: palette.err, backgroundColor: palette.errFill, fill: true },
        ],
      },
      options: {
        ...commonOpts,
        scales: {
          ...commonOpts.scales,
          yScore: { position: "left", min: 0, max: 100, ticks: { color: palette.tick }, grid: { color: palette.grid } },
          y01:    { position: "right", min: 0, max: 1, ticks: { color: palette.tick }, grid: { drawOnChartArea: false } },
          yErr:   { position: "right", min: 0, max: 8, ticks: { color: palette.tick }, grid: { drawOnChartArea: false } },
        }
      }
    });
  }


  function updateCharts() {
    const lead = state.charts.leading;
    const ops = state.charts.ops;
    if (!lead || !ops) return;

    // labels (time)
    state.chartLabels.push(fmtTime());
    while (state.chartLabels.length > state.chartWindow) state.chartLabels.shift();

    // push values
    const push = (arr, v) => { arr.push(v); while (arr.length > state.chartWindow) arr.shift(); };

    const ld = lead.data.datasets;
    push(ld[0].data, state.lagMs);
    push(ld[1].data, state.drift);
    push(ld[2].data, state.precursors);

    const od = ops.data.datasets;
    push(od[0].data, state.resilience);
    push(od[1].data, state.attackSuccess);
    push(od[2].data, state.errRate);

    lead.data.labels = state.chartLabels;
    ops.data.labels = state.chartLabels;

    lead.update("none");
    ops.update("none");
  }

  // -----------------------------
  // Render
  // -----------------------------
  function render() {
    // Executive summary
    setBadge("#riskBadge", state.riskLevel);
    setText("#riskReason", state.riskReason);
    setText("#resilienceScore", state.resilience);
    setProgress("#resilienceBar", state.resilience / 100);

    // Leading indicators (top)
    setText("#lag", Math.round(state.lagMs));
    setProgress("#lagBar", pct(state.lagMs, 100, 650));

    // CloudOps also shows lag (separate IDs)
    setText("#lagCloud", Math.round(state.lagMs));
    setProgress("#lagBarCloud", pct(state.lagMs, 100, 650));
    setBadge("#lagBadge", state._lagBadge || "STABLE");

    setText("#drift", state.drift.toFixed(2));
    setProgress("#driftBar", pct(state.drift, 0.05, 0.45));

    setText("#anom", state.precursors.toFixed(2));
    setProgress("#anomBar", pct(state.precursors, 0.05, 0.55));

    setText("#liPipelineTrend", state._trendLag || "↔");
    setText("#liDriftTrend", state._trendDrift || "↔");
    setText("#liAnomTrend", state._trendAnom || "↔");

    // Recommendations
    const recHTML = (state._recommendations || []).map((r) =>
      `<li class="flex gap-2"><span class="text-slate-400">•</span><span>${r}</span></li>`
    ).join("");
    setHTML("#recs", recHTML);

    // SOC
    setText("#lastUpdate", fmtTime());
    setText("#aCritical", state.alerts.critical);
    setText("#aHigh", state.alerts.high);
    setText("#aMedium", state.alerts.medium);
    setText("#aLow", state.alerts.low);

    setText("#ttd", state.ttd);
    setText("#ttr", state.ttr);

    // Gauges: lower is better (invert)
    setGauge("#ttdGauge", 1 - pct(state.ttd, 10, 180));
    setGauge("#ttrGauge", 1 - pct(state.ttr, 30, 420));

    // MITRE tags
    const tags = state.mitre.map((t) => `<span class="badge rounded-pill bg-dark me-1">${t}</span>`).join("");
    setHTML("#mitreTags", tags);

    // Feed
    const feedHTML = state.feed.map((e) => `
      <div class="timeline-item">
        <div class="t"><i class="bi ${e.icon} me-1"></i>${e.t}</div>
        <div class="m">${e.message}</div>
      </div>
    `).join("");
    setHTML("#feed", feedHTML);

    // MLOps
    setText("#conf", state.conf.toFixed(2));
    setProgress("#confBar", state.conf);

    setText("#fp", state.fpPer1k.toFixed(1));
    setProgress("#fpBar", pct(state.fpPer1k, 0.5, 10));

    setText("#advPressure", state.advPressure.toFixed(2));
    setProgress("#advBar", state.advPressure);

    setText("#attackSuccess", state.attackSuccess.toFixed(2));
    setProgress("#asrBar", state.attackSuccess);

    // Robustness badge
    const rb = $("#robustBadge");
    if (rb) {
      rb.textContent = state.robustLabel;
      rb.className =
        "badge rounded-pill " +
        (state.robustLabel === "STRONG" ? "bg-success" :
          state.robustLabel === "FAIR" ? "bg-warning text-dark" : "bg-danger");
    }

    setText("#modelVersion", state.modelVersion);
    setBadge("#retrainStatus", state.retrainStatus);

    // QuantumOps
    setText("#qQueue", state.qQueue);
    setText("#fidelity", state.fidelity.toFixed(2));
    setProgress("#fidelityBar", pct(state.fidelity, 0.80, 0.99));

    setText("#samplerLatency", Math.round(state.samplerLatency));
    setProgress("#samplerBar", pct(state.samplerLatency, 20, 160));

    setText("#qfeHealth", state.qfeHealth.toFixed(2));
    setProgress("#qfeBar", state.qfeHealth);

    // Latency split
    const qPart = Math.round(clamp(state.samplerLatency, 15, 180));
    const nPart = Math.round(clamp(state.lagMs * 0.22 + 20, 20, 220));
    setText("#latSplitQ", qPart);
    setText("#latSplitN", nPart);
    setProgress("#latSplitBarQ", pct(qPart, 15, 180));
    setProgress("#latSplitBarN", pct(nPart, 20, 220));
    setText("#latTotal", qPart + nPart);

    // CloudOps
    setText("#svcHealth", state.serviceHealth.toFixed(2));
    setProgress("#svcHealthBar", state.serviceHealth);

    setText("#p95", Math.round(state.p95));
    setProgress("#p95Bar", pct(state.p95, 120, 700));

    setText("#err", state.errRate.toFixed(1));
    setProgress("#errBar", pct(state.errRate, 0, 5));

    setText("#cost", state.costSpike.toFixed(2));
    setProgress("#costBar", pct(state.costSpike, 1.0, 1.35));

    // Regions
    const regionScore = (state.serviceHealth * 0.55 + (1 - pct(state.errRate, 0, 5)) * 0.25 + (1 - pct(state.p95, 120, 700)) * 0.20);
    const mk = (name, s) => `${name} • ${s > 0.72 ? "OK" : s > 0.55 ? "DEGRADED" : "IMPACTED"}`;
    const r1 = $("#r1"), r2 = $("#r2"), r3 = $("#r3");
    if (r1) r1.textContent = mk("us-east", clamp(regionScore + weightedJitter(0.04), 0, 1));
    if (r2) r2.textContent = mk("us-central", clamp(regionScore + weightedJitter(0.06), 0, 1));
    if (r3) r3.textContent = mk("eu-west", clamp(regionScore + weightedJitter(0.07), 0, 1));

    // Governance
    setText("#openRisks", state.openRisks);
    setText("#exceptions", state.exceptions);
    setText("#evidence", state.evidence);
    setText("#lastAudit", `${Math.round(state.lastAuditMin)}m`);

    // Sparklines
    drawSparkline($("#spLag"), state.hist.lag, { min: 80, max: 650 });
    drawSparkline($("#spDrift"), state.hist.drift, { min: 0.05, max: 0.45 });
    drawSparkline($("#spAnom"), state.hist.prec, { min: 0.05, max: 0.55 });
    drawSparkline($("#spAdv"), state.hist.adv, { min: 0.10, max: 0.90 });
    drawSparkline($("#spCost"), state.hist.cost, { min: 1.0, max: 1.35 });
    drawSparkline($("#spFid"), state.hist.fid, { min: 0.80, max: 0.99 });
  }

  // -----------------------------
  // Controls / actions
  // -----------------------------
  function setScenario(key) {
    state.scenario = key;
    setText("#scenarioName", SCENARIOS[key].name);
    setText("#scenarioDesc", SCENARIOS[key].desc);

    // Active button states (Bootstrap)
    const btns = {
      normal: $("#scenarioNormal"),
      adversarial: $("#scenarioAdversarial"),
      quantum: $("#scenarioQuantum"),
    };
    Object.entries(btns).forEach(([k, b]) => {
      if (!b) return;
      b.classList.toggle("btn-light", k === key);
      b.classList.toggle("btn-outline-light", k !== key);
    });

    addEvent(`Scenario switched to: ${SCENARIOS[key].name}`, "info");
  }

  function togglePause() {
    state.paused = !state.paused;
    const btn = $("#btnPause");
    if (btn) {
      btn.innerHTML = state.paused
        ? `<i class="bi bi-play-circle"></i> Resume`
        : `<i class="bi bi-pause-circle"></i> Pause`;
      btn.classList.toggle("btn-outline-warning", !state.paused);
      btn.classList.toggle("btn-outline-success", state.paused);
    }
    addEvent(state.paused ? "Telemetry paused." : "Telemetry resumed.", "info");
  }

  function resetDemo() {
    const s = SCENARIOS[state.scenario].dyn;

    state.lagMs = 120;
    state.drift = 0.12;
    state.precursors = 0.08;
    state.advPressure = 0.22;
    state.costSpike = 1.05;
    state.fidelity = s.fidelityTarget;

    state.serviceHealth = 0.96;
    state.p95 = 210;
    state.errRate = 0.6;
    state.qQueue = 2;
    state.samplerLatency = s.qLatencyTarget;
    state.qfeHealth = 0.91;

    state.conf = 0.93;
    state.fpPer1k = 2.1;
    state.attackSuccess = 0.07;

    state.ttd = 24;
    state.ttr = 68;

    state.openRisks = 5;
    state.exceptions = 2;
    state.evidence = 41;
    state.lastAuditMin = 12;

    state.quarantine = false;
    state.rateLimit = false;
    state.quantumBackoff = false;

    state.hist = { lag: [], drift: [], prec: [], adv: [], cost: [], fid: [] };
    state.feed = [];
    state.chartLabels = [];

    const tq = $("#toggleQuarantine"), tr = $("#toggleRateLimit"), tb = $("#toggleBackoff");
    if (tq) tq.checked = false;
    if (tr) tr.checked = false;
    if (tb) tb.checked = false;

    addEvent("Dashboard reset to baseline telemetry.", "ok");
  }

  function bumpVersion(v, dir) {
    const m = String(v).match(/^v(\d+)\.(\d+)\.(\d+)$/);
    if (!m) return v;
    let a = Number(m[1]), b = Number(m[2]), c = Number(m[3]);
    c = Math.max(0, c + dir);
    return `v${a}.${b}.${c}`;
  }

  function doRollback() {
    state.modelVersion = bumpVersion(state.modelVersion, -1);
    state.fpPer1k = clamp(state.fpPer1k * 0.86, 0.2, 25);
    state.attackSuccess = clamp(state.attackSuccess * 0.90, 0.005, 0.65);
    state.conf = clamp(state.conf + 0.04, 0.05, 0.99);
    state.lagMs = clamp(state.lagMs - 40, 60, 1200);
    addEvent(`Rollback executed: model now ${state.modelVersion}.`, "warn");

    computeRiskAndResilience();
    updateCharts();
    render();
  }

  function doRetrain() {
    state.retrainStatus = "Running";
    addEvent("Retraining trigger executed (simulated).", "info");

    // Retrain reduces drift/precursors, improves confidence, but costs resources briefly
    state.drift = clamp(state.drift * 0.75, 0.01, 0.65);
    state.precursors = clamp(state.precursors * 0.78, 0.01, 0.90);
    state.conf = clamp(state.conf + 0.03, 0.05, 0.99);
    state.costSpike = clamp(state.costSpike + 0.04, 0.90, 1.8);

    state.modelVersion = bumpVersion(state.modelVersion, +1);

    computeRiskAndResilience();
    updateCharts();
    render();

    setTimeout(() => {
      state.retrainStatus = "Ready";
      addEvent(`Retraining complete: deployed ${state.modelVersion}.`, "ok");
      computeRiskAndResilience();
      updateCharts();
      render();
    }, 1200);
  }

  function doAudit() {
    // Evidence capture should feel immediate (no waiting for the next tick)
    state.evidence += Math.round(1 + Math.random() * 3);
    state.lastAuditMin = 0;
    addEvent("Audit evidence captured and timestamped (simulated).", "ok");
    render();
  }

  function bind() {
    const bn = $("#scenarioNormal");
    const ba = $("#scenarioAdversarial");
    const bq = $("#scenarioQuantum");

    if (bn) bn.addEventListener("click", () => setScenario("normal"));
    if (ba) ba.addEventListener("click", () => setScenario("adversarial"));
    if (bq) bq.addEventListener("click", () => setScenario("quantum"));

    const bp = $("#btnPause");
    const br = $("#btnClearFeed");
    if (bp) bp.addEventListener("click", togglePause);
    if (br) br.addEventListener("click", resetDemo);

    const tq = $("#toggleQuarantine");
    const tr = $("#toggleRateLimit");
    const tb = $("#toggleBackoff");

    if (tq) tq.addEventListener("change", (e) => {
      state.quarantine = !!e.target.checked;
      addEvent(`Auto-quarantine ${state.quarantine ? "ENABLED" : "DISABLED"}.`, state.quarantine ? "warn" : "info");
    });
    if (tr) tr.addEventListener("change", (e) => {
      state.rateLimit = !!e.target.checked;
      addEvent(`Rate-limiting ${state.rateLimit ? "ENABLED" : "DISABLED"}.`, state.rateLimit ? "warn" : "info");
    });
    if (tb) tb.addEventListener("change", (e) => {
      state.quantumBackoff = !!e.target.checked;
      addEvent(`Quantum job backoff ${state.quantumBackoff ? "ENABLED" : "DISABLED"}.`, state.quantumBackoff ? "warn" : "info");
    });

    const rb = $("#btnRollback");
    const rt = $("#btnRetrain");
    const ra = $("#btnAudit");
    if (rb) rb.addEventListener("click", doRollback);
    if (rt) rt.addEventListener("click", doRetrain);
    if (ra) ra.addEventListener("click", doAudit);
  }

  // -----------------------------
  // Main loop
  // -----------------------------
  function tick() {
    if (!state.paused) {
      applyScenarioForces();
      computeRiskAndResilience();
      updateAlerts();
      updateCharts();
    }
    render();
  }

  function init() {
    setScenario("normal");

    addEvent("Dashboard initialized: mock telemetry engine online.", "ok");
    addEvent("Leading indicators drive risk/resilience before failures occur.", "info");

    bind();
    initCharts();

    // initial compute & render
    computeRiskAndResilience();
    render();
    updateCharts();

    setInterval(tick, 1000);
  }

  window.addEventListener("DOMContentLoaded", init);
})();
