from __future__ import annotations
import torch
import torch.nn.functional as F

@torch.no_grad()
def clamp_like_standardized(x: torch.Tensor, minv: float = -4.0, maxv: float = 4.0) -> torch.Tensor:
    # Since inputs are standardized, keep within a reasonable range.
    return torch.clamp(x, minv, maxv)

def fgsm_attack(model, x: torch.Tensor, y: torch.Tensor, eps: float, angle_sigma: float, output_sigma: float):
    x_adv = x.clone().detach().requires_grad_(True)
    logits = model(x_adv, angle_sigma=angle_sigma, output_sigma=output_sigma)
    loss = F.cross_entropy(logits, y)
    loss.backward()
    grad = x_adv.grad.detach().sign()
    x_adv = x_adv.detach() + eps * grad
    x_adv = clamp_like_standardized(x_adv)
    return x_adv

def pgd_attack(model, x: torch.Tensor, y: torch.Tensor, eps: float, alpha: float, steps: int, angle_sigma: float, output_sigma: float):
    x0 = x.clone().detach()
    x_adv = x0 + 0.001 * torch.randn_like(x0)
    x_adv = clamp_like_standardized(x_adv)
    for _ in range(steps):
        x_adv.requires_grad_(True)
        logits = model(x_adv, angle_sigma=angle_sigma, output_sigma=output_sigma)
        loss = F.cross_entropy(logits, y)
        loss.backward()
        grad = x_adv.grad.detach().sign()
        x_adv = x_adv.detach() + alpha * grad

        # project back to eps-ball
        delta = torch.clamp(x_adv - x0, -eps, eps)
        x_adv = clamp_like_standardized(x0 + delta)
    return x_adv
