from __future__ import annotations
import numpy as np
import torch
import pennylane as qml

def make_device(n_qubits: int):
    # default.qubit is fast and supports autograd through Torch interface
    return qml.device("default.qubit", wires=n_qubits)

def vqc_expectations(
    x_angles: torch.Tensor,
    weights: torch.Tensor,
    n_qubits: int,
    output_sigma: float = 0.0,
) -> torch.Tensor:
    """Compute expectation values from a variational quantum circuit (VQC).

    - x_angles: [batch, n_qubits] angles for encoding
    - weights:  [n_layers, n_qubits, 3] trainable rotation params
    Returns:    [batch, n_qubits] expectations (Z on each qubit)
    """
    dev = make_device(n_qubits)

    @qml.qnode(dev, interface="torch", diff_method="parameter-shift")
    def circuit(x, w):
        # angle encoding
        for i in range(n_qubits):
            qml.RY(x[i], wires=i)
            qml.RZ(0.5 * x[i], wires=i)

        # variational layers
        for l in range(w.shape[0]):
            for i in range(n_qubits):
                qml.RX(w[l, i, 0], wires=i)
                qml.RY(w[l, i, 1], wires=i)
                qml.RZ(w[l, i, 2], wires=i)
            # entanglement ring
            for i in range(n_qubits - 1):
                qml.CNOT(wires=[i, i + 1])
            qml.CNOT(wires=[n_qubits - 1, 0])

        return [qml.expval(qml.PauliZ(i)) for i in range(n_qubits)]

    # vectorize over batch
    outs = []
    for b in range(x_angles.shape[0]):
        outs.append(torch.stack(circuit(x_angles[b], weights)))
    out = torch.stack(outs)  # [batch, n_qubits]

    if output_sigma and output_sigma > 0:
        out = out + torch.randn_like(out) * output_sigma

    return out

def inject_angle_noise(x_angles: torch.Tensor, sigma: float) -> torch.Tensor:
    if sigma <= 0:
        return x_angles
    return x_angles + torch.randn_like(x_angles) * sigma
